﻿using System;
using System.Collections;
using System.Collections.Generic;
using Unity.Mathematics;
using UnityEngine;

public class GameController : MonoBehaviour
{
    protected class BufferedAction
    {
        public bool firstFrame = true;
        public string name;
        public string[] stringArray;
        public Action callback;
    }

    public static GameController instance;

    public AudioSource audioSource;
    public AudioSource interactSfx;

    public int thresholdMovement1 = 10;
    public int thresholdMovement2 = 20;
    public int thresholdDrunk = 50;

    protected bool silenced = false;
    protected HashSet<string> flags = new HashSet<string>() {};
    protected List<Tuple<int, string>> restrictions = new List<Tuple<int, string>>();
    protected int hazardCounter = 0;

    protected List<BufferedAction> bufferedActions = new List<BufferedAction>();

    public readonly System.Random random = new System.Random();

    private void OnEnable()
    {
        instance = this;

        ScoreBoard.Instance.flags = flags;
    }

    void Update()
    {
        audioSource.volume = Mathf.Lerp(audioSource.volume, silenced ? 0f : .5f, 0.05f);

        while (bufferedActions.Count > 0 && !bufferedActions[0].firstFrame)
        {
            switch (bufferedActions[0].name)
            {
                case "StartDialog":
                    DialogController.instance.Activate(bufferedActions[0].stringArray, bufferedActions[0].callback);
                    break;

                case "StopDialog":
                    DialogController.instance.Deactivate();
                    break;

                case "StartCabinetMinigame":
                    Player.instance.Deactivate();
                    TooltipController.instance.Deactivate();
                    IngredientsCabinetController.instance.Activate();
                    break;

                case "StopCabinetMinigame":
                    IngredientsCabinetController.instance.Deactivate();
                    Player.instance.Activate();
                    break;

                case "StartElectricOvenMinigame":
                    Player.instance.Deactivate();
                    TooltipController.instance.Deactivate();
                    ElectricOvenController.instance.Activate();
                    break;

                case "StopElectricOvenMinigame":
                    ElectricOvenController.instance.Deactivate();
                    Player.instance.Activate();
                    break;

                case "StartConfirmExitDialogController":
                    Player.instance.Deactivate();
                    TooltipController.instance.Deactivate();
                    ConfirmExitDialogController.instance.Activate();
                    break;

                case "StopConfirmExitDialogController":
                    ConfirmExitDialogController.instance.Deactivate();
                    Player.instance.Activate();
                    break;
            }

            bufferedActions.RemoveAt(0);
        }

        foreach (BufferedAction ba in bufferedActions)
        {
            ba.firstFrame = false;
        }
    }

    public bool HasFlag(string flag)
    {
        return flags.Contains(flag);
    }

    public void AddFlag(string flag, GameObject[] objectsToActivate=null, GameObject[] objectsToDeactivate=null)
    {
        if (flag != "")
        {
            if (!flags.Contains(flag))
            {
                Debug.Log("Got Flag: " + flag);
            }

            flags.Add(flag);
        }

        if (objectsToActivate != null)
        {
            foreach (GameObject go in objectsToActivate)
            {
                go.SetActive(true);
            }
        }

        if (objectsToDeactivate != null)
        {
            foreach (GameObject go in objectsToDeactivate)
            {
                go.SetActive(false);
            }
        }
    }

    public void RemoveFlag(string flag)
    {
        if (flags.Contains(flag))
        {
            Debug.Log("Lost Flag: " + flag);
        }

        flags.Remove(flag);
    }

    public bool HasRestriction(string restriction)
    {
        foreach (Tuple<int, string> t in restrictions) {
            if (t.Item2 == restriction)
            {
                return true;
            }
        }

        return false;
    }

    public void ResetHazardCounter()
    {
        restrictions.Clear();
        hazardCounter = 0;

        StartDialog(new string[] { "You feel like you are fully in control again." });
    }

    public void DecreaseHazardCounter(int step)
    {
        bool regained = false;

        if (hazardCounter >= thresholdMovement1 && hazardCounter - step < thresholdMovement1)
        {
            for (int i=0; i<restrictions.Count; ++i)
            {
                if (restrictions[i].Item1 == 1)
                {
                    Debug.Log("Removed restriction: " + restrictions[i].Item2);
                    restrictions.RemoveAt(i);
                    regained = true;
                    break;
                }
            }
        }

        if (hazardCounter >= thresholdMovement2 && hazardCounter - step < thresholdMovement2)
        {
            for (int i=0; i<restrictions.Count; ++i)
            {
                if (restrictions[i].Item1 == 2)
                {
                    Debug.Log("Removed restriction: " + restrictions[i].Item2);
                    restrictions.RemoveAt(i);
                    regained = true;
                    break;
                }
            }
        }

        if (hazardCounter >= thresholdDrunk && hazardCounter - step < thresholdDrunk)
        {
            for (int i=0; i<restrictions.Count; ++i)
            {
                if (restrictions[i].Item1 == 3)
                {
                    Debug.Log("Removed restriction: " + restrictions[i].Item2);
                    restrictions.RemoveAt(i);
                    regained = true;
                    break;
                }
            }
        }

        if (regained)
        {
            StartDialog(new string[] { "You feel like you have regained some control." });
        }

        hazardCounter = Math.Max(0, hazardCounter - step);
        Debug.Log("Hazard Counter: " + hazardCounter);
    }

    public void IncrementHazardCounter(int step=1)
    {
        if (hazardCounter < thresholdMovement1 && hazardCounter + step >= thresholdMovement1)
        {
            StartDialog(new string[] { "Richard: What's wrong with me, it feels like I can't control myself.", "Richard is wrestling for control, your movement is impaired." });
            switch (random.Next(4))
            {
                case 0:
                    Debug.Log("Add restriction Up");
                    restrictions.Add(new Tuple<int, string>(1, "Up"));
                    break;
                case 1:
                    Debug.Log("Add restriction Right");
                    restrictions.Add(new Tuple<int, string>(1, "Right"));
                    break;
                case 2:
                    Debug.Log("Add restriction Down");
                    restrictions.Add(new Tuple<int, string>(1, "Down"));
                    break;
                case 3:
                    Debug.Log("Add restriction Left");
                    restrictions.Add(new Tuple<int, string>(1, "Left"));
                    break;
            }
        }

        if (hazardCounter < thresholdMovement2 && hazardCounter + step >= thresholdMovement2)
        {
            StartDialog(new string[] { "Richard: What's happening? I need to stop this." });
            switch (random.Next(2))
            {
                case 0:
                    {
                        string restriction = HasRestriction("Up") || HasRestriction("Down") ? "Right" : "Up";
                        Debug.Log("Add restriction " + restriction);
                        restrictions.Add(new Tuple<int, string>(2, restriction));
                        break;
                    }
                case 1:
                    {
                        string restriction = HasRestriction("Up") || HasRestriction("Down") ? "Left" : "Down";
                        Debug.Log("Add restriction " + restriction);
                        restrictions.Add(new Tuple<int, string>(2, restriction));
                        break;
                    }
            }
        }

        if (hazardCounter < thresholdDrunk && hazardCounter + step >= thresholdDrunk)
        {
            Debug.Log("Add restriction Drunk");
            StartDialog(new string[] { "Richard: I must be possessed or something, I'll stop this at once." });
            restrictions.Add(new Tuple<int, string>(3, "Drunk"));
        }

        hazardCounter = Math.Min(Math.Max(Math.Max(thresholdMovement1, thresholdMovement2), thresholdDrunk) + 5, hazardCounter + step);
    }

    public void PlayInteractSFX()
    {
        interactSfx.Play();
    }

    public void StartDialog(string[] dialogTexts, Action callback = null)
    {
        if (dialogTexts.Length > 0)
        {
            bufferedActions.Add(new BufferedAction() { name = "StartDialog", stringArray = dialogTexts, callback = callback });
        } else
        {
            callback?.Invoke();
        }
    }

    public void StopDialog()
    {
        bufferedActions.Add(new BufferedAction() { name = "StopDialog" });
    }

    public void StartPushup()
    {
        silenced = true;
        Player.instance.animator.SetBool("IsMoving", false);
        Player.instance.Deactivate();
        TooltipController.instance.Deactivate();
        PushupMinigameController.instance.Activate();
    }

    public void StopPushup()
    {
        PushupMinigameController.instance.Deactivate();
        Player.instance.Activate();
        silenced = false;
    }

    public void StartCabinetMinigame()
    {
        bufferedActions.Add(new BufferedAction() { name = "StartCabinetMinigame" });
        silenced = true;
    }

    public void StopCabinetMinigame()
    {
        bufferedActions.Add(new BufferedAction() { name = "StopCabinetMinigame" });
        silenced = false;
    }

    public void StartElectricOvenMinigame()
    {
        silenced = true;
        bufferedActions.Add(new BufferedAction() { name = "StartElectricOvenMinigame" });
    }

    public void StopElectricOvenMinigame()
    {
        silenced = false;
        bufferedActions.Add(new BufferedAction() { name = "StopElectricOvenMinigame" });
    }

    public void StartConfirmExitDialogController()
    {
        bufferedActions.Add(new BufferedAction() { name = "StartConfirmExitDialogController" });
    }

    public void StopConfirmExitDialogController()
    {
        bufferedActions.Add(new BufferedAction() { name = "StopConfirmExitDialogController" });
    }
}
